	function hdlspec = hdlfir(name, bitspec)
	%hdlspec = hdlfir(name, bitspec)
	%
	%Generates an HDL specification for an FIR filter. The number of 
	%components generated is algorithm- and architecture-dependent.
	%
	%The main components are:
	%  name - Structural component for the FIR filter. The interface
	%    consists of:
	%      clk, reset: Inputs of type std_logic
	%      x_0 .. x_{bitspec.num_phases-1}: Inputs of type
	%        std_logic_vector(bitspec.wdata-1 downto 0)
	%      y_0 .. y_{bitspec.num_branches-1}: Outputs of type
	%        std_logic_vector(bitspec.wout-1 downto 0)
	%  name_ppc - partial product generation that is shared between branches
	%  name_pp%d - branch-specific partial product generation
	%  name_sha - adders shared between branches
	%  name_cs%d - reduction tree
	%  name_vma%d - VMA
	%
	%Arguments:
	%  name - name of generated filter
	%  bitspec - specification structure
	%
	%Returns:
	%  hdlspec - HDL specification structure
	
	%Copyright 2008, 2010 Anton Blad
	%
	%This file is part of firgen.
	%
	%firgen is free software: you can redistribute it and/or modify
	%it under the terms of the GNU General Public License as published by
	%the Free Software Foundation, either version 3 of the License, or
	%(at your option) any later version.
	%
	%firgen is distributed in the hope that it will be useful,
	%but WITHOUT ANY WARRANTY; without even the implied warranty of
	%MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	%GNU General Public License for more details.
	%
	%You should have received a copy of the GNU General Public License
	%along with firgen.  If not, see <http://www.gnu.org/licenses/>
	
	hdlspec.top = name;
	hdlspec.components = [];
	hdlspec.components = [hdlspec.components, hdlcompiface('fa')];
	hdlspec.components = [hdlspec.components, hdlcompiface('ha')];
	hdlspec.components = [hdlspec.components, hdlcompiface('fa_noc')];
	hdlspec.components = [hdlspec.components, hdlcompiface('ha_noc')];
	hdlspec.components = [hdlspec.components, hdlcompiface('dff')];
	
	for kin = 0:bitspec.numin-1
		strucinputs(kin+1) = hdlsignal(sprintf('x_%d', kin), 1, bitspec.arith.wdata);
	end
	for kout = 0:bitspec.numout-1
		strucoutputs(kout+1) = hdlsignal(sprintf('y_%d', kout), 1, bitspec.arith.wout);
	end
	struccomp = hdlstruc(name, [], strucinputs, strucoutputs);
	
	pptype = bitspec.branches{1}.ppmap.type;
	
	if strcmp(pptype, 'DF')
		delays = zeros(1, bitspec.numin);
		for b = 1:bitspec.numout
			delays = max(delays, bitspec.branches{b}.ppmap.delays);
		end
	
		hdlspec.components = [hdlspec.components, hdlcompiface('reg')];
	
		ppccomp = hdlppdfc(sprintf('%s_ppc', name), bitspec.numin, bitspec.arith.wdata, delays);
	elseif strcmp(pptype, 'TF')
		ppccomp = hdlpptfc(sprintf('%s_ppc', name), bitspec.numin, bitspec.arith.wdata);
	end
	
	hdlspec.components = [hdlspec.components, ppccomp];
	[struccomp, ppcid] = hdladdinstance(struccomp, ppccomp);
	struccomp = hdlconnect(struccomp, 'in', 'x', ppcid, 'x');
	
	if isfield(bitspec, 'shared')
		shacomp = hdlsha(sprintf('%s_sha', name), ppccomp.iface.outputs, bitspec.shared);
		hdlspec.components = [hdlspec.components, shacomp];
		[struccomp, shaid] = hdladdinstance(struccomp, shacomp);
		struccomp = hdlconnect(struccomp, ppcid, 'xc', shaid, 'xc');
		ppinputs = [ppccomp.iface.outputs, shacomp.iface.outputs];
	else
		ppinputs = ppccomp.iface.outputs;
	end
	
	for b = 0:bitspec.numout-1
		if strcmp(pptype, 'DF')
			ppcomp = hdlppdf(sprintf('%s_pp%d', name, b), ppinputs, bitspec.branches{b+1}.ppmap.bits);
			hdlspec.components = [hdlspec.components, ppcomp];
			[struccomp, ppid] = hdladdinstance(struccomp, ppcomp);
			struccomp = hdlconnect(struccomp, ppcid, 'xc', ppid, 'xc');
			ppids{b+1} = ppid;
		elseif strcmp(pptype, 'TF')
			ppcomp = hdlpptf(sprintf('%s_pp%d', name, b), ppinputs, bitspec.branches{b+1}.ppmap.bits);
			hdlspec.components = [hdlspec.components, ppcomp];
			[struccomp, ppid] = hdladdinstance(struccomp, ppcomp);
			struccomp = hdlconnect(struccomp, ppcid, 'xc', ppid, 'xc');
			ppids{b+1} = ppid;
		end
		if isfield(bitspec, 'shared')
			struccomp = hdlconnect(struccomp, shaid, 'sha', ppid, 'sha');
		end
	end
	
	for b = 0:bitspec.numout-1
		cscomp = hdlcs(sprintf('%s_cs%d', name, b), bitspec.branches{b+1}.csspec, bitspec.branches{b+1}.csmap);
		vmacomp = hdlvmacs(sprintf('%s_vma%d', name, b), bitspec.branches{b+1}.vmaspec, bitspec.branches{b+1}.vmamap);
		hdlspec.components = [hdlspec.components, cscomp, vmacomp];
		[struccomp, csid] = hdladdinstance(struccomp, cscomp);
		[struccomp, vmaid] = hdladdinstance(struccomp, vmacomp);
		struccomp = hdlconnect(struccomp, ppids{b+1}, 'pp', csid, 'in');
		struccomp = hdlconnect(struccomp, csid, 'out', vmaid, 'in_1');
		struccomp = hdlconnect(struccomp, vmaid, 'y', 'out', sprintf('y_%d', b));
	end
	
	hdlspec.components = [hdlspec.components, struccomp];
	
